// multi-chain-wallet.js
// Robust, Multi-Chain Wallet Logic with Fallback & Validation

class MultiChainWallet {
    constructor(config) {
        this.config = config;
        this.providers = {}; // Cache providers
        this.currentChain = 'ethereum';
        
        // Ensure libraries are available
        if (typeof ethers === 'undefined') console.error('Ethers.js is missing!');
        // Check for Solana Web3 if needed
    }

    // --- 1. Connection & RPC Fallback System ---

    async getProvider(chainKey) {
        const chain = this.config.CHAINS[chainKey];
        if (!chain) throw new Error(`Chain ${chainKey} not configured`);

        // Return cached provider if working
        if (this.providers[chainKey]) {
            try {
                if (chain.type === 'evm') {
                    await this.providers[chainKey].getBlockNumber();
                } else if (chain.type === 'solana') {
                    await this.providers[chainKey].getVersion();
                }
                return this.providers[chainKey];
            } catch (e) {
                console.warn(`Cached provider for ${chainKey} failed, reconnecting...`);
            }
        }

        // Fallback Logic
        console.log(`Connecting to ${chain.name}...`);
        for (const url of chain.rpcs) {
            try {
                let provider;
                if (chain.type === 'evm') {
                    provider = new ethers.JsonRpcProvider(url);
                    await provider.getBlockNumber(); // Test connection
                } else if (chain.type === 'solana' && window.solanaWeb3) {
                    provider = new window.solanaWeb3.Connection(url, 'confirmed');
                    await provider.getVersion(); // Test connection
                } else if (chain.type === 'bitcoin') {
                    // For Bitcoin, "provider" is just the base API URL for REST calls
                    const res = await fetch(`${url}/blocks/tip/height`);
                    if (!res.ok) throw new Error('API failed');
                    provider = { baseUrl: url }; // Custom object for REST
                } else {
                    // Generic/Others
                    provider = { url }; 
                }

                console.log(`Connected to ${chain.name} via ${url}`);
                this.providers[chainKey] = provider;
                return provider;
            } catch (e) {
                console.warn(`RPC ${url} failed for ${chain.name}:`, e.message);
            }
        }
        
        throw new Error(`All RPCs failed for ${chain.name}. Please check your internet connection.`);
    }

    // --- 2. Address Validation ---

    validateAddress(address, chainKey) {
        const chain = this.config.CHAINS[chainKey];
        if (!address) return { valid: false, message: 'Address is empty' };

        try {
            if (chain.type === 'evm') {
                const isValid = ethers.isAddress(address);
                return { valid: isValid, message: isValid ? 'Valid EVM Address' : 'Invalid EVM Address' };
            } 
            else if (chain.type === 'solana' && window.solanaWeb3) {
                try {
                    new window.solanaWeb3.PublicKey(address);
                    return { valid: true, message: 'Valid Solana Address' };
                } catch {
                    return { valid: false, message: 'Invalid Solana Address' };
                }
            }
            else if (chain.type === 'bitcoin') {
                // Basic Regex for BTC (Legacy, Segwit, Taproot)
                const btcRegex = /^(bc1|[13])[a-zA-HJ-NP-Z0-9]{25,39}$/;
                const isValid = btcRegex.test(address);
                return { valid: isValid, message: isValid ? 'Valid Bitcoin Address' : 'Invalid Bitcoin Format' };
            }
            
            return { valid: true, message: 'Format check skipped' }; // Default allow
        } catch (e) {
            return { valid: false, message: 'Validation Error' };
        }
    }

    // --- 3. Real-time Gas Estimation ---

    async estimateFee(chainKey, to, amount) {
        const chain = this.config.CHAINS[chainKey];
        const provider = await this.getProvider(chainKey);

        try {
            if (chain.type === 'evm') {
                const feeData = await provider.getFeeData();
                // Standard transfer gas limit
                const gasLimit = 21000n; 
                const gasPrice = feeData.gasPrice || feeData.maxFeePerGas;
                const totalFeeWei = gasLimit * gasPrice;
                return {
                    native: ethers.formatEther(totalFeeWei),
                    symbol: chain.nativeCurrency.symbol,
                    raw: totalFeeWei
                };
            } 
            else if (chain.type === 'solana') {
                // Solana fee is usually fixed (5000 lamports) per signature
                return {
                    native: '0.000005', // approx
                    symbol: 'SOL',
                    raw: 5000
                };
            }
            else if (chain.type === 'bitcoin') {
                // Fetch fees from mempool.space
                const res = await fetch(`${provider.baseUrl}/v1/fees/recommended`);
                const fees = await res.json();
                const vBytes = 140; // Approx for simple tx
                const satoshis = fees.fastestFee * vBytes;
                return {
                    native: (satoshis / 100000000).toFixed(8),
                    symbol: 'BTC',
                    raw: satoshis
                };
            }
            
            return { native: '0.00', symbol: '', raw: 0 };
        } catch (e) {
            console.error('Fee estimation failed:', e);
            throw new Error('Failed to estimate gas');
        }
    }

    // --- 4. Sending Transactions (Router) ---

    async sendTransaction(chainKey, seedOrSigner, to, amount) {
        const chain = this.config.CHAINS[chainKey];
        
        if (chain.type === 'evm') {
            return this.sendEVMTransaction(chainKey, seedOrSigner, to, amount);
        } else if (chain.type === 'solana') {
            throw new Error('Solana sending requires wallet adapter (Phantom/Solflare) integration.');
        } else {
            throw new Error(`${chain.name} sending not fully implemented in this demo.`);
        }
    }

    async sendEVMTransaction(chainKey, seed, to, amount) {
        const provider = await this.getProvider(chainKey);
        // Securely create wallet from seed
        const wallet = ethers.HDNodeWallet.fromPhrase(seed).connect(provider);
        
        const tx = await wallet.sendTransaction({
            to: to,
            value: ethers.parseEther(amount)
        });
        
        return {
            hash: tx.hash,
            wait: () => tx.wait()
        };
    }

    // --- 5. Transaction Listener ---
    
    async listenForTransaction(chainKey, address, callback) {
        const chain = this.config.CHAINS[chainKey];
        const provider = await this.getProvider(chainKey);
        
        console.log(`Listening on ${chain.name} for ${address}`);
        
        let lastBalance = -1n;

        const check = async () => {
            try {
                if (chain.type === 'evm') {
                    const bal = await provider.getBalance(address);
                    if (lastBalance === -1n) {
                        lastBalance = bal;
                    } else if (bal > lastBalance) {
                        const diff = bal - lastBalance;
                        callback({
                            amount: ethers.formatEther(diff),
                            symbol: chain.nativeCurrency.symbol
                        });
                        lastBalance = bal;
                    }
                }
                // Add other chain listeners here
            } catch (e) {
                console.warn('Listener poll failed', e);
            }
        };

        // Poll every 5s
        setInterval(check, 5000);
        check(); // Initial
    }
}

// Global instance helper
window.WalletCore = new MultiChainWallet(CONFIG);
